// Admin Panel JavaScript for Joveh Hub

class AdminPanel {
    constructor() {
        this.init();
    }

    init() {
        this.initSidebar();
        this.initTables();
        this.initForms();
        this.initModals();
        this.initCharts();
        this.initFileUpload();
        this.initUserManagement();
        this.initReports();
    }

    initSidebar() {
        // Mobile sidebar toggle
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebar = document.querySelector('.admin-sidebar');

        if (sidebarToggle && sidebar) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('collapsed');
                document.querySelector('.admin-main').classList.toggle('expanded');
            });
        }

        // Active menu highlighting
        const currentPath = window.location.pathname;
        const navItems = document.querySelectorAll('.nav-item');

        navItems.forEach(item => {
            const href = item.getAttribute('href');
            if (href && currentPath.includes(href)) {
                item.classList.add('active');
            }
        });

        // Submenu toggle
        const hasSubmenu = document.querySelectorAll('.has-submenu');
        hasSubmenu.forEach(item => {
            item.addEventListener('click', (e) => {
                if (window.innerWidth < 1024) {
                    e.preventDefault();
                    const submenu = item.nextElementSibling;
                    submenu.classList.toggle('show');
                }
            });
        });
    }

    initTables() {
        // Sortable tables
        const sortableHeaders = document.querySelectorAll('.sortable');
        sortableHeaders.forEach(header => {
            header.addEventListener('click', () => {
                const table = header.closest('table');
                const tbody = table.querySelector('tbody');
                const rows = Array.from(tbody.querySelectorAll('tr'));
                const index = Array.from(header.parentNode.children).indexOf(header);
                const isAsc = header.classList.contains('asc');

                rows.sort((a, b) => {
                    const aText = a.children[index].textContent.trim();
                    const bText = b.children[index].textContent.trim();

                    if (isNaN(aText) || isNaN(bText)) {
                        return isAsc ?
                            bText.localeCompare(aText) :
                            aText.localeCompare(bText);
                    } else {
                        return isAsc ?
                            parseFloat(bText) - parseFloat(aText) :
                            parseFloat(aText) - parseFloat(bText);
                    }
                });

                // Remove all rows
                while (tbody.firstChild) {
                    tbody.removeChild(tbody.firstChild);
                }

                // Append sorted rows
                rows.forEach(row => tbody.appendChild(row));

                // Update sort indicators
                sortableHeaders.forEach(h => h.classList.remove('asc', 'desc'));
                header.classList.toggle('asc', !isAsc);
                header.classList.toggle('desc', isAsc);
            });
        });

        // Search functionality
        const searchInputs = document.querySelectorAll('.table-search');
        searchInputs.forEach(input => {
            input.addEventListener('input', debounce((e) => {
                const searchTerm = e.target.value.toLowerCase();
                const table = e.target.closest('.table-container').querySelector('table');
                const rows = table.querySelectorAll('tbody tr');

                rows.forEach(row => {
                    const text = row.textContent.toLowerCase();
                    row.style.display = text.includes(searchTerm) ? '' : 'none';
                });
            }, 300));
        });

        // Pagination
        const paginationLinks = document.querySelectorAll('.pagination a');
        paginationLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                // In a real implementation, this would fetch new data
                // For now, just update active state
                paginationLinks.forEach(l => l.classList.remove('active'));
                link.classList.add('active');
            });
        });
    }

    initForms() {
        // Form validation
        const forms = document.querySelectorAll('form');
        forms.forEach(form => {
            form.addEventListener('submit', (e) => {
                let isValid = true;
                const requiredFields = form.querySelectorAll('[required]');

                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        isValid = false;
                        this.showFieldError(field, 'این فیلد الزامی است');
                    } else {
                        this.clearFieldError(field);
                    }

                    // Email validation
                    if (field.type === 'email') {
                        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        if (!emailRegex.test(field.value)) {
                            isValid = false;
                            this.showFieldError(field, 'ایمیل وارد شده معتبر نیست');
                        }
                    }

                    // Phone validation
                    if (field.type === 'tel' || field.name.includes('phone')) {
                        const phoneRegex = /^[0-9]{10,11}$/;
                        if (!phoneRegex.test(field.value)) {
                            isValid = false;
                            this.showFieldError(field, 'شماره تماس معتبر نیست');
                        }
                    }

                    // National code validation
                    if (field.name.includes('national_code')) {
                        if (!this.validateNationalCode(field.value)) {
                            isValid = false;
                            this.showFieldError(field, 'کد ملی معتبر نیست');
                        }
                    }
                });

                if (!isValid) {
                    e.preventDefault();
                    this.showNotification('لطفا خطاهای فرم را برطرف کنید', 'error');
                }
            });
        });

        // Real-time validation
        const realTimeFields = document.querySelectorAll('input[required], textarea[required], select[required]');
        realTimeFields.forEach(field => {
            field.addEventListener('blur', () => {
                if (!field.value.trim()) {
                    this.showFieldError(field, 'این فیلد الزامی است');
                } else {
                    this.clearFieldError(field);
                }
            });
        });

        // Character counters
        const textareas = document.querySelectorAll('textarea[maxlength]');
        textareas.forEach(textarea => {
            const maxLength = parseInt(textarea.getAttribute('maxlength'));
            const counter = document.createElement('div');
            counter.className = 'char-counter';
            counter.style.fontSize = '0.75rem';
            counter.style.color = '#6c757d';
            counter.style.textAlign = 'left';
            counter.style.marginTop = '5px';
            textarea.parentNode.appendChild(counter);

            const updateCounter = () => {
                const currentLength = textarea.value.length;
                counter.textContent = `${currentLength}/${maxLength}`;

                if (currentLength > maxLength * 0.9) {
                    counter.style.color = '#f39c12';
                } else if (currentLength > maxLength) {
                    counter.style.color = '#e74c3c';
                } else {
                    counter.style.color = '#6c757d';
                }
            };

            textarea.addEventListener('input', updateCounter);
            updateCounter();
        });
    }

    initModals() {
        // Modal triggers
        const modalTriggers = document.querySelectorAll('[data-modal]');
        modalTriggers.forEach(trigger => {
            trigger.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = trigger.getAttribute('data-modal');
                const modal = document.getElementById(modalId);

                if (modal) {
                    this.openModal(modal);
                }
            });
        });

        // Modal close buttons
        const modalCloses = document.querySelectorAll('.modal-close, .modal-cancel');
        modalCloses.forEach(close => {
            close.addEventListener('click', () => {
                const modal = close.closest('.modal');
                this.closeModal(modal);
            });
        });

        // Close modal on outside click
        document.addEventListener('click', (e) => {
            const modals = document.querySelectorAll('.modal.show');
            modals.forEach(modal => {
                if (e.target === modal) {
                    this.closeModal(modal);
                }
            });
        });

        // Close modal on escape key
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                const modals = document.querySelectorAll('.modal.show');
                modals.forEach(modal => this.closeModal(modal));
            }
        });
    }

    initCharts() {
        // Initialize charts if Chart.js is available
        if (typeof Chart !== 'undefined') {
            this.initUserChart();
            this.initNotesChart();
            this.initReportsChart();
        }
    }

    initUserChart() {
        const ctx = document.getElementById('user-chart');
        if (ctx) {
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: ['فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور'],
                    datasets: [{
                        label: 'کاربران جدید',
                        data: [12, 19, 8, 15, 22, 30],
                        borderColor: '#4361ee',
                        backgroundColor: 'rgba(67, 97, 238, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            labels: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        },
                        x: {
                            ticks: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    initNotesChart() {
        const ctx = document.getElementById('notes-chart');
        if (ctx) {
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: ['حسابان', 'فیزیک', 'شیمی', 'هندسه', 'آمار'],
                    datasets: [{
                        label: 'تعداد جزوات',
                        data: [15, 12, 8, 10, 6],
                        backgroundColor: [
                            'rgba(67, 97, 238, 0.7)',
                            'rgba(46, 204, 113, 0.7)',
                            'rgba(155, 89, 182, 0.7)',
                            'rgba(241, 196, 15, 0.7)',
                            'rgba(231, 76, 60, 0.7)'
                        ],
                        borderColor: [
                            '#4361ee',
                            '#2ecc71',
                            '#9b59b6',
                            '#f1c40f',
                            '#e74c3c'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            labels: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        },
                        x: {
                            ticks: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    initReportsChart() {
        const ctx = document.getElementById('reports-chart');
        if (ctx) {
            new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: ['در انتظار', 'مشاهده شده', 'حل شده'],
                    datasets: [{
                        data: [5, 8, 12],
                        backgroundColor: [
                            '#ffc107',
                            '#17a2b8',
                            '#28a745'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                font: {
                                    family: 'Vazirmatn'
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    initFileUpload() {
        // Drag and drop
        const dropZones = document.querySelectorAll('.drop-zone');
        dropZones.forEach(zone => {
            zone.addEventListener('dragover', (e) => {
                e.preventDefault();
                zone.classList.add('dragover');
            });

            zone.addEventListener('dragleave', () => {
                zone.classList.remove('dragover');
            });

            zone.addEventListener('drop', (e) => {
                e.preventDefault();
                zone.classList.remove('dragover');

                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    const fileInput = zone.querySelector('input[type="file"]');
                    if (fileInput) {
                        // Create a new FileList (can't set directly, need to use DataTransfer)
                        const dataTransfer = new DataTransfer();
                        dataTransfer.items.add(files[0]);
                        fileInput.files = dataTransfer.files;

                        // Trigger change event
                        fileInput.dispatchEvent(new Event('change', { bubbles: true }));
                    }
                }
            });
        });

        // File preview for images
        const imageUploads = document.querySelectorAll('input[type="file"][accept*="image"]');
        imageUploads.forEach(input => {
            input.addEventListener('change', (e) => {
                const file = e.target.files[0];
                if (file && file.type.startsWith('image/')) {
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        const preview = input.parentElement.querySelector('.image-preview') ||
                            this.createImagePreview(input.parentElement);
                        preview.innerHTML = `<img src="${e.target.result}" alt="Preview">`;
                        preview.style.display = 'block';
                    };
                    reader.readAsDataURL(file);
                }
            });
        });
    }

    initUserManagement() {
        // Bulk actions
        const bulkSelect = document.querySelector('.bulk-select');
        const bulkActions = document.querySelector('.bulk-actions');
        const userCheckboxes = document.querySelectorAll('.user-checkbox');

        if (bulkSelect && bulkActions && userCheckboxes.length > 0) {
            bulkSelect.addEventListener('change', (e) => {
                userCheckboxes.forEach(checkbox => {
                    checkbox.checked = e.target.checked;
                });
                bulkActions.style.display = e.target.checked ? 'block' : 'none';
            });

            userCheckboxes.forEach(checkbox => {
                checkbox.addEventListener('change', () => {
                    const checkedCount = document.querySelectorAll('.user-checkbox:checked').length;
                    bulkSelect.checked = checkedCount === userCheckboxes.length;
                    bulkActions.style.display = checkedCount > 0 ? 'block' : 'none';
                });
            });

            // Bulk delete
            const bulkDelete = document.querySelector('.bulk-delete');
            if (bulkDelete) {
                bulkDelete.addEventListener('click', () => {
                    const checkedIds = Array.from(document.querySelectorAll('.user-checkbox:checked'))
                        .map(checkbox => checkbox.value);

                    if (checkedIds.length > 0 && confirm(`آیا از حذف ${checkedIds.length} کاربر انتخاب شده مطمئن هستید؟`)) {
                        // In real implementation, send AJAX request
                        this.showNotification(`${checkedIds.length} کاربر با موفقیت حذف شدند`, 'success');
                    }
                });
            }
        }

        // User search
        const userSearch = document.getElementById('user-search');
        if (userSearch) {
            userSearch.addEventListener('input', debounce((e) => {
                const searchTerm = e.target.value.toLowerCase();
                const rows = document.querySelectorAll('#users-table tbody tr');

                rows.forEach(row => {
                    const name = row.querySelector('.user-name').textContent.toLowerCase();
                    const nationalCode = row.querySelector('.user-national-code').textContent.toLowerCase();
                    const email = row.querySelector('.user-email')?.textContent.toLowerCase() || '';

                    row.style.display = name.includes(searchTerm) ||
                        nationalCode.includes(searchTerm) ||
                        email.includes(searchTerm) ? '' : 'none';
                });
            }, 300));
        }
    }

    initReports() {
        // Report status filters
        const statusFilters = document.querySelectorAll('.status-filter');
        statusFilters.forEach(filter => {
            filter.addEventListener('click', (e) => {
                e.preventDefault();
                const status = filter.getAttribute('data-status');

                // Update active filter
                statusFilters.forEach(f => f.classList.remove('active'));
                filter.classList.add('active');

                // Filter reports
                const rows = document.querySelectorAll('#reports-table tbody tr');
                rows.forEach(row => {
                    const rowStatus = row.querySelector('.report-status').textContent.trim();
                    if (status === 'all' || rowStatus === status) {
                        row.style.display = '';
                    } else {
                        row.style.display = 'none';
                    }
                });
            });
        });

        // Mark as resolved
        const resolveButtons = document.querySelectorAll('.resolve-report');
        resolveButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const reportId = button.getAttribute('data-id');

                if (confirm('آیا این گزارش حل شده است؟')) {
                    // In real implementation, send AJAX request
                    button.closest('tr').querySelector('.report-status').textContent = 'resolved';
                    button.closest('tr').querySelector('.report-status').className = 'report-status status-resolved';
                    button.remove();

                    this.showNotification('گزارش به عنوان حل‌شده علامت‌گذاری شد', 'success');
                }
            });
        });
    }

    // Utility methods
    showFieldError(field, message) {
        this.clearFieldError(field);

        const error = document.createElement('div');
        error.className = 'field-error';
        error.textContent = message;
        error.style.color = '#e74c3c';
        error.style.fontSize = '0.875rem';
        error.style.marginTop = '5px';

        field.parentNode.appendChild(error);
        field.classList.add('error');
    }

    clearFieldError(field) {
        const existingError = field.parentNode.querySelector('.field-error');
        if (existingError) {
            existingError.remove();
        }
        field.classList.remove('error');
    }

    validateNationalCode(code) {
        if (!/^\d{10}$/.test(code)) return false;

        const check = parseInt(code[9]);
        let sum = 0;

        for (let i = 0; i < 9; i++) {
            sum += parseInt(code[i]) * (10 - i);
        }

        let remainder = sum % 11;
        if (remainder < 2) {
            return check === remainder;
        } else {
            return check === 11 - remainder;
        }
    }

    openModal(modal) {
        modal.classList.add('show');
        document.body.style.overflow = 'hidden';

        // Focus first input
        const input = modal.querySelector('input, textarea, select');
        if (input) input.focus();
    }

    closeModal(modal) {
        modal.classList.remove('show');
        document.body.style.overflow = '';
    }

    createImagePreview(container) {
        const preview = document.createElement('div');
        preview.className = 'image-preview';
        preview.style.marginTop = '10px';
        preview.style.display = 'none';

        const img = document.createElement('img');
        img.style.maxWidth = '200px';
        img.style.maxHeight = '200px';
        img.style.borderRadius = '8px';
        img.style.boxShadow = '0 2px 4px rgba(0,0,0,0.1)';

        preview.appendChild(img);
        container.appendChild(preview);

        return preview;
    }

    showNotification(message, type = 'info') {
        // Use main.js notification system if available
        if (typeof window.JovehHub !== 'undefined' &&
            typeof window.JovehHub.showNotification === 'function') {
            window.JovehHub.showNotification(message, type);
        } else {
            // Fallback notification
            alert(message);
        }
    }
}

// Helper functions
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.adminPanel = new AdminPanel();
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = AdminPanel;
}