// Main JavaScript File for Joveh Hub
(function () {
    'use strict';

    // Main Application
    class JovehHubApp {
        constructor() {
            this.init();
        }

        init() {
            console.log('Joveh Hub App Initialized');
            this.initNavigation();
            this.initForms();
            this.initSecurity();
            this.initAnimations();
            this.initNotifications();
            this.initUIComponents();
        }

        // Navigation functionality
        initNavigation() {
            // Mobile menu toggle
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const mobileMenu = document.querySelector('.mobile-menu');

            if (mobileMenuBtn && mobileMenu) {
                mobileMenuBtn.addEventListener('click', () => {
                    mobileMenu.classList.toggle('active');
                    mobileMenuBtn.classList.toggle('active');
                });
            }

            // Dropdown menus
            const dropdownToggles = document.querySelectorAll('.dropdown-toggle');
            dropdownToggles.forEach(toggle => {
                toggle.addEventListener('click', (e) => {
                    e.preventDefault();
                    const dropdown = toggle.closest('.dropdown');
                    const menu = dropdown.querySelector('.dropdown-menu');

                    if (menu) {
                        menu.classList.toggle('show');
                    }
                });
            });

            // Close dropdowns when clicking outside
            document.addEventListener('click', (e) => {
                if (!e.target.closest('.dropdown')) {
                    document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                        menu.classList.remove('show');
                    });
                }
            });

            // User menu dropdown
            const userInfo = document.querySelector('.user-info');
            if (userInfo) {
                userInfo.addEventListener('click', (e) => {
                    e.stopPropagation();
                    const menu = document.querySelector('.menu-dropdown');
                    if (menu) {
                        menu.classList.toggle('show');
                    }
                });

                // Close user menu when clicking outside
                document.addEventListener('click', () => {
                    const menu = document.querySelector('.menu-dropdown.show');
                    if (menu) {
                        menu.classList.remove('show');
                    }
                });
            }
        }

        // Form validation and enhancement
        initForms() {
            // Password strength indicator
            const passwordInputs = document.querySelectorAll('input[type="password"]');
            passwordInputs.forEach(input => {
                input.addEventListener('input', () => {
                    this.checkPasswordStrength(input);
                });
            });

            // Form validation
            const forms = document.querySelectorAll('form');
            forms.forEach(form => {
                form.addEventListener('submit', (e) => {
                    if (!this.validateForm(form)) {
                        e.preventDefault();
                        this.showNotification('لطفا تمام فیلدهای ضروری را پر کنید', 'error');
                    }
                });
            });

            // File upload preview
            const fileInputs = document.querySelectorAll('input[type="file"]');
            fileInputs.forEach(input => {
                input.addEventListener('change', (e) => {
                    this.updateFilePreview(e.target);
                });
            });

            // Real-time validation
            const realTimeFields = document.querySelectorAll('[required]');
            realTimeFields.forEach(field => {
                field.addEventListener('blur', () => {
                    this.validateField(field);
                });
            });
        }

        // Check password strength
        checkPasswordStrength(input) {
            const password = input.value;
            const strengthIndicator = input.parentElement.querySelector('.password-strength');

            if (!strengthIndicator) return;

            let strength = 0;
            let text = '';

            // Check criteria
            if (password.length >= 8) strength++;
            if (/[a-z]/.test(password)) strength++;
            if (/[A-Z]/.test(password)) strength++;
            if (/[0-9]/.test(password)) strength++;
            if (/[^a-zA-Z0-9]/.test(password)) strength++;

            // Determine strength level
            switch (strength) {
                case 0:
                case 1:
                    text = 'خیلی ضعیف';
                    strengthIndicator.className = 'password-strength weak';
                    break;
                case 2:
                    text = 'ضعیف';
                    strengthIndicator.className = 'password-strength moderate';
                    break;
                case 3:
                    text = 'متوسط';
                    strengthIndicator.className = 'password-strength good';
                    break;
                case 4:
                    text = 'قوی';
                    strengthIndicator.className = 'password-strength strong';
                    break;
                case 5:
                    text = 'خیلی قوی';
                    strengthIndicator.className = 'password-strength very-strong';
                    break;
            }

            strengthIndicator.textContent = text;
        }

        // Validate form
        validateForm(form) {
            const requiredFields = form.querySelectorAll('[required]');
            let isValid = true;

            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    isValid = false;
                    this.showFieldError(field, 'این فیلد الزامی است');
                } else {
                    this.clearFieldError(field);
                }

                // Email validation
                if (field.type === 'email') {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(field.value)) {
                        isValid = false;
                        this.showFieldError(field, 'ایمیل وارد شده معتبر نیست');
                    }
                }

                // Phone validation
                if (field.type === 'tel' || field.name.includes('phone')) {
                    const phoneRegex = /^[0-9]{10,11}$/;
                    if (!phoneRegex.test(field.value)) {
                        isValid = false;
                        this.showFieldError(field, 'شماره تماس معتبر نیست');
                    }
                }
            });

            return isValid;
        }

        // Validate single field
        validateField(field) {
            if (!field.value.trim()) {
                this.showFieldError(field, 'این فیلد الزامی است');
            } else {
                this.clearFieldError(field);
            }
        }

        // Show field error
        showFieldError(field, message) {
            this.clearFieldError(field);

            const error = document.createElement('div');
            error.className = 'field-error';
            error.textContent = message;
            error.style.cssText = `
                color: #e74c3c;
                font-size: 0.875rem;
                margin-top: 5px;
                display: block;
            `;

            field.parentNode.appendChild(error);
            field.classList.add('error');
        }

        // Clear field error
        clearFieldError(field) {
            const existingError = field.parentNode.querySelector('.field-error');
            if (existingError) {
                existingError.remove();
            }
            field.classList.remove('error');
        }

        // Update file preview
        updateFilePreview(input) {
            const fileInfo = input.parentElement.querySelector('.file-upload-info');
            if (!fileInfo || !input.files || input.files.length === 0) return;

            const file = input.files[0];
            const fileSize = (file.size / (1024 * 1024)).toFixed(2);
            const fileName = file.name.length > 20 ?
                file.name.substring(0, 20) + '...' : file.name;

            fileInfo.innerHTML = `
                <i class="fas fa-file-pdf"></i>
                <span>${fileName}</span>
                <small>${fileSize} MB</small>
            `;
        }

        // Security features
        initSecurity() {
            // Prevent right-click in PDF viewer
            if (window.location.pathname.includes('/notes/view.php')) {
                document.addEventListener('contextmenu', (e) => {
                    e.preventDefault();
                    this.showNotification('امکان راست کلیک در این صفحه غیرفعال است', 'warning');
                });
            }

            // Prevent keyboard shortcuts
            document.addEventListener('keydown', (e) => {
                // Prevent F12
                if (e.key === 'F12') {
                    e.preventDefault();
                    return false;
                }

                // Prevent Ctrl+S, Ctrl+P, Ctrl+C
                if (e.ctrlKey) {
                    if (e.key === 's' || e.key === 'p' || e.key === 'c') {
                        e.preventDefault();
                        this.showNotification('این عملیات مجاز نیست', 'warning');
                        return false;
                    }
                }

                // Prevent DevTools shortcuts
                if (e.ctrlKey && e.shiftKey) {
                    if (e.key === 'I' || e.key === 'J' || e.key === 'C') {
                        e.preventDefault();
                        return false;
                    }
                }
            });

            // Prevent text selection in PDF viewer
            const pdfCanvas = document.getElementById('pdf-canvas');
            if (pdfCanvas) {
                pdfCanvas.addEventListener('selectstart', (e) => {
                    e.preventDefault();
                    return false;
                });
            }

            // Disable drag and drop for PDF files
            if (window.location.pathname.includes('/notes/view.php')) {
                document.addEventListener('dragstart', (e) => {
                    if (e.target.closest('#pdf-canvas')) {
                        e.preventDefault();
                    }
                });
            }
        }

        // Animations
        initAnimations() {
            // Smooth scrolling
            const smoothScrollLinks = document.querySelectorAll('a[href^="#"]');
            smoothScrollLinks.forEach(link => {
                link.addEventListener('click', (e) => {
                    const targetId = link.getAttribute('href');
                    if (targetId === '#') return;

                    const targetElement = document.querySelector(targetId);
                    if (targetElement) {
                        e.preventDefault();
                        window.scrollTo({
                            top: targetElement.offsetTop - 100,
                            behavior: 'smooth'
                        });
                    }
                });
            });

            // Scroll animations
            const observerOptions = {
                root: null,
                rootMargin: '0px',
                threshold: 0.1
            };

            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-in');
                    }
                });
            }, observerOptions);

            const animatedElements = document.querySelectorAll('.feature, .card, .subject-category');
            animatedElements.forEach(element => {
                observer.observe(element);
            });

            // Add animation styles
            this.addAnimationStyles();
        }

        // Add animation styles
        addAnimationStyles() {
            const style = document.createElement('style');
            style.textContent = `
                .animate-in {
                    animation: fadeInUp 0.6s ease-out forwards;
                }

                @keyframes fadeInUp {
                    from {
                        opacity: 0;
                        transform: translateY(30px);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0);
                    }
                }

                /* Password strength styles */
                .password-strength {
                    font-size: 0.75rem;
                    margin-top: 5px;
                    padding: 3px 8px;
                    border-radius: 4px;
                    display: inline-block;
                }

                .password-strength.weak {
                    background-color: #f8d7da;
                    color: #721c24;
                }

                .password-strength.moderate {
                    background-color: #fff3cd;
                    color: #856404;
                }

                .password-strength.good {
                    background-color: #d1ecf1;
                    color: #0c5460;
                }

                .password-strength.strong {
                    background-color: #d4edda;
                    color: #155724;
                }

                .password-strength.very-strong {
                    background-color: #c3e6cb;
                    color: #155724;
                    font-weight: bold;
                }

                /* Error styles */
                input.error,
                select.error,
                textarea.error {
                    border-color: #e74c3c !important;
                }

                .field-error {
                    color: #e74c3c;
                    font-size: 0.875rem;
                    margin-top: 5px;
                    display: block;
                }

                /* Dropdown styles */
                .dropdown-menu {
                    display: none;
                    position: absolute;
                    background: white;
                    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                    border-radius: 8px;
                    z-index: 1000;
                }

                .dropdown-menu.show {
                    display: block;
                }

                .menu-dropdown.show {
                    display: block !important;
                }
            `;
            document.head.appendChild(style);
        }

        // Notification system
        initNotifications() {
            // Add notification styles
            const notificationStyle = document.createElement('style');
            notificationStyle.textContent = `
                .notification {
                    position: fixed;
                    top: 20px;
                    right: 20px;
                    padding: 15px 20px;
                    border-radius: 8px;
                    display: flex;
                    align-items: center;
                    gap: 10px;
                    z-index: 9999;
                    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                    animation: slideIn 0.3s ease-out;
                    max-width: 400px;
                }

                .notification-success {
                    background-color: #d4edda;
                    border: 1px solid #c3e6cb;
                    color: #155724;
                }

                .notification-error {
                    background-color: #f8d7da;
                    border: 1px solid #f5c6cb;
                    color: #721c24;
                }

                .notification-warning {
                    background-color: #fff3cd;
                    border: 1px solid #ffeeba;
                    color: #856404;
                }

                .notification-info {
                    background-color: #d1ecf1;
                    border: 1px solid #bee5eb;
                    color: #0c5460;
                }

                .notification-icon {
                    font-weight: bold;
                }

                .notification-close {
                    background: none;
                    border: none;
                    font-size: 1.5rem;
                    cursor: pointer;
                    padding: 0;
                    margin-right: auto;
                    opacity: 0.7;
                }

                .notification-close:hover {
                    opacity: 1;
                }

                @keyframes slideIn {
                    from {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }

                @keyframes slideOut {
                    from {
                        transform: translateX(0);
                        opacity: 1;
                    }
                    to {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                }
            `;
            document.head.appendChild(notificationStyle);
        }

        // Show notification
        showNotification(message, type = 'info') {
            // Remove existing notifications
            const existingNotifications = document.querySelectorAll('.notification');
            existingNotifications.forEach(notification => {
                notification.remove();
            });

            // Create notification element
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;

            const icon = type === 'success' ? '✓' :
                type === 'error' ? '✕' :
                    type === 'warning' ? '⚠' : 'ℹ';

            notification.innerHTML = `
                <span class="notification-icon">${icon}</span>
                <span class="notification-message">${message}</span>
                <button class="notification-close">&times;</button>
            `;

            // Add close button functionality
            const closeBtn = notification.querySelector('.notification-close');
            closeBtn.addEventListener('click', () => {
                notification.style.animation = 'slideOut 0.3s ease-out';
                setTimeout(() => notification.remove(), 300);
            });

            // Auto remove after 5 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.style.animation = 'slideOut 0.3s ease-out';
                    setTimeout(() => notification.remove(), 300);
                }
            }, 5000);

            document.body.appendChild(notification);
        }

        // Initialize UI components
        initUIComponents() {
            // Tooltips
            this.initTooltips();

            // Tabs
            this.initTabs();

            // Accordions
            this.initAccordions();

            // Modal dialogs
            this.initModals();

            // Toast notifications
            this.initToasts();

            // Progress bars
            this.initProgressBars();
        }

        initTooltips() {
            const tooltips = document.querySelectorAll('[data-tooltip]');
            tooltips.forEach(element => {
                element.addEventListener('mouseenter', (e) => {
                    const tooltipText = element.getAttribute('data-tooltip');
                    const tooltip = document.createElement('div');
                    tooltip.className = 'tooltip';
                    tooltip.textContent = tooltipText;
                    tooltip.style.cssText = `
                        position: absolute;
                        background: #333;
                        color: white;
                        padding: 5px 10px;
                        border-radius: 4px;
                        font-size: 0.75rem;
                        z-index: 10000;
                        white-space: nowrap;
                        pointer-events: none;
                    `;

                    document.body.appendChild(tooltip);

                    const rect = element.getBoundingClientRect();
                    tooltip.style.top = (rect.top - tooltip.offsetHeight - 5) + 'px';
                    tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';

                    element.tooltipElement = tooltip;
                });

                element.addEventListener('mouseleave', () => {
                    if (element.tooltipElement) {
                        element.tooltipElement.remove();
                        element.tooltipElement = null;
                    }
                });
            });
        }

        initTabs() {
            const tabContainers = document.querySelectorAll('.tab-container');
            tabContainers.forEach(container => {
                const tabButtons = container.querySelectorAll('.tab-button');
                const tabContents = container.querySelectorAll('.tab-content');

                tabButtons.forEach(button => {
                    button.addEventListener('click', () => {
                        const tabId = button.getAttribute('data-tab');

                        // Update buttons
                        tabButtons.forEach(btn => btn.classList.remove('active'));
                        button.classList.add('active');

                        // Update contents
                        tabContents.forEach(content => {
                            content.classList.remove('active');
                            if (content.id === tabId) {
                                content.classList.add('active');
                            }
                        });
                    });
                });
            });
        }

        initAccordions() {
            const accordionButtons = document.querySelectorAll('.accordion-button');
            accordionButtons.forEach(button => {
                button.addEventListener('click', () => {
                    const accordion = button.closest('.accordion');
                    const content = accordion.querySelector('.accordion-content');

                    const isActive = accordion.classList.contains('active');

                    // Close all accordions
                    document.querySelectorAll('.accordion.active').forEach(acc => {
                        acc.classList.remove('active');
                        const accContent = acc.querySelector('.accordion-content');
                        if (accContent) {
                            accContent.style.maxHeight = null;
                        }
                    });

                    // Open current if was closed
                    if (!isActive) {
                        accordion.classList.add('active');
                        if (content) {
                            content.style.maxHeight = content.scrollHeight + 'px';
                        }
                    }
                });
            });
        }

        initModals() {
            const modalTriggers = document.querySelectorAll('[data-modal]');
            modalTriggers.forEach(trigger => {
                trigger.addEventListener('click', (e) => {
                    e.preventDefault();
                    const modalId = trigger.getAttribute('data-modal');
                    const modal = document.getElementById(modalId);

                    if (modal) {
                        this.openModal(modal);
                    }
                });
            });

            const modalCloses = document.querySelectorAll('.modal-close, .modal-cancel');
            modalCloses.forEach(close => {
                close.addEventListener('click', () => {
                    const modal = close.closest('.modal');
                    this.closeModal(modal);
                });
            });

            document.addEventListener('click', (e) => {
                if (e.target.classList.contains('modal')) {
                    this.closeModal(e.target);
                }
            });

            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape') {
                    const modals = document.querySelectorAll('.modal.show');
                    modals.forEach(modal => this.closeModal(modal));
                }
            });
        }

        openModal(modal) {
            modal.classList.add('show');
            document.body.style.overflow = 'hidden';

            const input = modal.querySelector('input, textarea, select');
            if (input) input.focus();
        }

        closeModal(modal) {
            modal.classList.remove('show');
            document.body.style.overflow = '';
        }

        initToasts() {
            // Toast container
            let toastContainer = document.querySelector('.toast-container');
            if (!toastContainer) {
                toastContainer = document.createElement('div');
                toastContainer.className = 'toast-container';
                toastContainer.style.cssText = `
                    position: fixed;
                    top: 20px;
                    left: 20px;
                    z-index: 9998;
                `;
                document.body.appendChild(toastContainer);
            }
        }

        initProgressBars() {
            const progressBars = document.querySelectorAll('.progress-bar');
            progressBars.forEach(bar => {
                const percent = bar.getAttribute('data-percent');
                if (percent) {
                    bar.style.width = percent + '%';

                    // Animate on scroll
                    const observer = new IntersectionObserver((entries) => {
                        entries.forEach(entry => {
                            if (entry.isIntersecting) {
                                bar.classList.add('animate');
                            }
                        });
                    }, { threshold: 0.5 });

                    observer.observe(bar);
                }
            });
        }

        // Utility functions
        debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        throttle(func, limit) {
            let inThrottle;
            return function () {
                const args = arguments;
                const context = this;
                if (!inThrottle) {
                    func.apply(context, args);
                    inThrottle = true;
                    setTimeout(() => inThrottle = false, limit);
                }
            };
        }

        // Validate national code (Iranian)
        validateNationalCode(code) {
            if (!/^\d{10}$/.test(code)) return false;

            const check = parseInt(code[9]);
            let sum = 0;

            for (let i = 0; i < 9; i++) {
                sum += parseInt(code[i]) * (10 - i);
            }

            let remainder = sum % 11;
            if (remainder < 2) {
                return check === remainder;
            } else {
                return check === 11 - remainder;
            }
        }

        // Format date
        formatDate(date, format = 'YYYY/MM/DD') {
            const d = new Date(date);
            const year = d.getFullYear();
            const month = String(d.getMonth() + 1).padStart(2, '0');
            const day = String(d.getDate()).padStart(2, '0');

            return format
                .replace('YYYY', year)
                .replace('MM', month)
                .replace('DD', day);
        }

        // AJAX helper
        async ajaxRequest(url, options = {}) {
            const defaultOptions = {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: null
            };

            const mergedOptions = { ...defaultOptions, ...options };

            try {
                const response = await fetch(url, mergedOptions);

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const contentType = response.headers.get('content-type');
                if (contentType && contentType.includes('application/json')) {
                    return await response.json();
                } else {
                    return await response.text();
                }
            } catch (error) {
                console.error('AJAX request failed:', error);
                this.showNotification('خطا در ارتباط با سرور', 'error');
                throw error;
            }
        }
    }

    // Initialize app when DOM is loaded
    document.addEventListener('DOMContentLoaded', () => {
        window.jovehHub = new JovehHubApp();

        // Expose utility functions globally
        window.debounce = window.jovehHub.debounce.bind(window.jovehHub);
        window.throttle = window.jovehHub.throttle.bind(window.jovehHub);
        window.showNotification = window.jovehHub.showNotification.bind(window.jovehHub);

        console.log('Joveh Hub Application Loaded Successfully');
    });

    // Add global error handler
    window.addEventListener('error', (event) => {
        console.error('Global error:', event.error);

        // Don't show error notifications for production
        if (window.location.hostname === 'localhost' ||
            window.location.hostname === '127.0.0.1') {
            window.jovehHub?.showNotification(`خطا: ${event.error.message}`, 'error');
        }
    });

    // Add unhandled promise rejection handler
    window.addEventListener('unhandledrejection', (event) => {
        console.error('Unhandled promise rejection:', event.reason);
    });

})();

// Export for module systems (if needed)
if (typeof module !== 'undefined' && module.exports) {
    module.exports = window.jovehHub;
}