// PDF Viewer Component for Joveh Hub
class PDFViewer {
    constructor(pdfUrl, options = {}) {
        this.pdfUrl = pdfUrl;
        this.options = {
            scale: 1.5,
            maxScale: 3,
            minScale: 0.5,
            scaleStep: 0.25,
            ...options
        };

        this.pdfDoc = null;
        this.pageNum = 1;
        this.pageRendering = false;
        this.pageNumPending = null;
        this.canvas = document.getElementById('pdf-canvas');
        this.ctx = this.canvas.getContext('2d');

        this.init();
    }

    async init() {
        try {
            // Load PDF document
            const loadingTask = pdfjsLib.getDocument(this.pdfUrl);
            this.pdfDoc = await loadingTask.promise;

            // Update page info
            document.getElementById('total-pages').textContent = this.pdfDoc.numPages;

            // Render first page
            await this.renderPage(this.pageNum);

            // Initialize controls
            this.initControls();

            // Hide loading indicator
            document.querySelector('.pdf-loading').style.display = 'none';

        } catch (error) {
            console.error('Error loading PDF:', error);
            document.querySelector('.pdf-loading').innerHTML = `
                <i class="fas fa-exclamation-circle"></i>
                <p>خطا در بارگذاری جزوه</p>
                <small>${error.message}</small>
            `;
        }
    }

    async renderPage(num) {
        this.pageRendering = true;

        try {
            // Get page
            const page = await this.pdfDoc.getPage(num);

            // Set canvas dimensions
            const viewport = page.getViewport({ scale: this.options.scale });
            this.canvas.height = viewport.height;
            this.canvas.width = viewport.width;

            // Render PDF page into canvas context
            const renderContext = {
                canvasContext: this.ctx,
                viewport: viewport
            };

            await page.render(renderContext).promise;

            // Update page info
            document.getElementById('current-page').textContent = num;

            this.pageRendering = false;

            // Render pending page if any
            if (this.pageNumPending !== null) {
                await this.renderPage(this.pageNumPending);
                this.pageNumPending = null;
            }

        } catch (error) {
            console.error('Error rendering page:', error);
            this.pageRendering = false;
        }
    }

    queueRenderPage(num) {
        if (this.pageRendering) {
            this.pageNumPending = num;
        } else {
            this.renderPage(num);
        }
    }

    initControls() {
        // Previous page
        document.getElementById('prev-page').addEventListener('click', () => {
            if (this.pageNum <= 1) return;
            this.pageNum--;
            this.queueRenderPage(this.pageNum);
        });

        // Next page
        document.getElementById('next-page').addEventListener('click', () => {
            if (this.pageNum >= this.pdfDoc.numPages) return;
            this.pageNum++;
            this.queueRenderPage(this.pageNum);
        });

        // Zoom in
        document.getElementById('zoom-in').addEventListener('click', () => {
            if (this.options.scale < this.options.maxScale) {
                this.options.scale += this.options.scaleStep;
                this.renderPage(this.pageNum);
            }
        });

        // Zoom out
        document.getElementById('zoom-out').addEventListener('click', () => {
            if (this.options.scale > this.options.minScale) {
                this.options.scale -= this.options.scaleStep;
                this.renderPage(this.pageNum);
            }
        });

        // Fullscreen
        document.getElementById('fullscreen').addEventListener('click', () => {
            const wrapper = document.getElementById('pdf-viewer-wrapper');

            if (!document.fullscreenElement) {
                if (wrapper.requestFullscreen) {
                    wrapper.requestFullscreen();
                } else if (wrapper.webkitRequestFullscreen) {
                    wrapper.webkitRequestFullscreen();
                } else if (wrapper.msRequestFullscreen) {
                    wrapper.msRequestFullscreen();
                }
            } else {
                if (document.exitFullscreen) {
                    document.exitFullscreen();
                } else if (document.webkitExitFullscreen) {
                    document.webkitExitFullscreen();
                } else if (document.msExitFullscreen) {
                    document.msExitFullscreen();
                }
            }
        });

        // Keyboard navigation
        document.addEventListener('keydown', (e) => {
            // Only handle keys when PDF viewer is focused
            if (!this.canvas.parentElement.contains(document.activeElement)) {
                return;
            }

            switch (e.key) {
                case 'ArrowLeft':
                case 'PageUp':
                    e.preventDefault();
                    if (this.pageNum > 1) {
                        this.pageNum--;
                        this.queueRenderPage(this.pageNum);
                    }
                    break;

                case 'ArrowRight':
                case 'PageDown':
                case ' ':
                    e.preventDefault();
                    if (this.pageNum < this.pdfDoc.numPages) {
                        this.pageNum++;
                        this.queueRenderPage(this.pageNum);
                    }
                    break;

                case '+':
                case '=':
                    e.preventDefault();
                    if (this.options.scale < this.options.maxScale) {
                        this.options.scale += this.options.scaleStep;
                        this.renderPage(this.pageNum);
                    }
                    break;

                case '-':
                    e.preventDefault();
                    if (this.options.scale > this.options.minScale) {
                        this.options.scale -= this.options.scaleStep;
                        this.renderPage(this.pageNum);
                    }
                    break;

                case 'Home':
                    e.preventDefault();
                    this.pageNum = 1;
                    this.queueRenderPage(this.pageNum);
                    break;

                case 'End':
                    e.preventDefault();
                    this.pageNum = this.pdfDoc.numPages;
                    this.queueRenderPage(this.pageNum);
                    break;
            }
        });

        // Touch gestures for mobile
        let touchStartX = 0;
        let touchStartY = 0;

        this.canvas.addEventListener('touchstart', (e) => {
            touchStartX = e.touches[0].clientX;
            touchStartY = e.touches[0].clientY;
        }, { passive: true });

        this.canvas.addEventListener('touchend', (e) => {
            if (!touchStartX || !touchStartY) return;

            const touchEndX = e.changedTouches[0].clientX;
            const touchEndY = e.changedTouches[0].clientY;

            const diffX = touchStartX - touchEndX;
            const diffY = touchStartY - touchEndY;

            // Horizontal swipe for page navigation
            if (Math.abs(diffX) > Math.abs(diffY)) {
                if (diffX > 50 && this.pageNum < this.pdfDoc.numPages) {
                    // Swipe left to go to next page
                    this.pageNum++;
                    this.queueRenderPage(this.pageNum);
                } else if (diffX < -50 && this.pageNum > 1) {
                    // Swipe right to go to previous page
                    this.pageNum--;
                    this.queueRenderPage(this.pageNum);
                }
            }

            // Vertical swipe for zoom
            if (Math.abs(diffY) > Math.abs(diffX)) {
                if (diffY > 50 && this.options.scale > this.options.minScale) {
                    // Swipe up to zoom out
                    this.options.scale -= this.options.scaleStep;
                    this.renderPage(this.pageNum);
                } else if (diffY < -50 && this.options.scale < this.options.maxScale) {
                    // Swipe down to zoom in
                    this.options.scale += this.options.scaleStep;
                    this.renderPage(this.pageNum);
                }
            }

            touchStartX = 0;
            touchStartY = 0;
        }, { passive: true });

        // Double tap to zoom
        let lastTap = 0;
        this.canvas.addEventListener('touchend', (e) => {
            const currentTime = new Date().getTime();
            const tapLength = currentTime - lastTap;

            if (tapLength < 300 && tapLength > 0) {
                // Double tap detected
                if (this.options.scale === 1.5) {
                    this.options.scale = 2.5;
                } else {
                    this.options.scale = 1.5;
                }
                this.renderPage(this.pageNum);
                e.preventDefault();
            }

            lastTap = currentTime;
        });

        // Page jump input
        const pageInfo = document.getElementById('page-info');
        pageInfo.addEventListener('click', () => {
            const pageNum = prompt('به کدام صفحه بروید؟', this.pageNum);
            if (pageNum) {
                const num = parseInt(pageNum);
                if (num >= 1 && num <= this.pdfDoc.numPages) {
                    this.pageNum = num;
                    this.queueRenderPage(this.pageNum);
                } else {
                    alert(`لطفا عددی بین ۱ تا ${this.pdfDoc.numPages} وارد کنید`);
                }
            }
        });

        // Print protection
        window.addEventListener('beforeprint', (e) => {
            e.preventDefault();
            alert('چاپ این جزوه مجاز نیست.');
            return false;
        });

        // Copy protection
        document.addEventListener('copy', (e) => {
            if (this.canvas.contains(e.target) ||
                document.getElementById('pdf-viewer-wrapper').contains(e.target)) {
                e.preventDefault();
                alert('کپی‌برداری از این جزوه مجاز نیست.');
            }
        });
    }

    // Public methods
    goToPage(num) {
        if (num >= 1 && num <= this.pdfDoc.numPages) {
            this.pageNum = num;
            this.queueRenderPage(num);
        }
    }

    zoomIn() {
        if (this.options.scale < this.options.maxScale) {
            this.options.scale += this.options.scaleStep;
            this.renderPage(this.pageNum);
        }
    }

    zoomOut() {
        if (this.options.scale > this.options.minScale) {
            this.options.scale -= this.options.scaleStep;
            this.renderPage(this.pageNum);
        }
    }

    resetZoom() {
        this.options.scale = 1.5;
        this.renderPage(this.pageNum);
    }

    // Get current page as image (for preview only)
    getCurrentPageAsImage() {
        return this.canvas.toDataURL('image/png');
    }
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = PDFViewer;
} else {
    window.PDFViewer = PDFViewer;
}

// Auto-initialize if data-pdf attribute exists
document.addEventListener('DOMContentLoaded', () => {
    const pdfElement = document.querySelector('[data-pdf]');
    if (pdfElement) {
        const pdfUrl = pdfElement.getAttribute('data-pdf');
        const options = JSON.parse(pdfElement.getAttribute('data-options') || '{}');
        new PDFViewer(pdfUrl, options);
    }
});