<?php
// جلوگیری از حملات XSS
function xss_clean($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    return $data;
}

// جلوگیری از SQL Injection
function sql_escape($data) {
    global $pdo;
    return $pdo->quote($data);
}

// اعتبارسنجی فایل آپلود شده
function validate_uploaded_file($file, $allowed_types = ['pdf'], $max_size = 52428800) {
    $errors = [];
    
    // بررسی وجود فایل
    if (!isset($file) || $file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'خطا در آپلود فایل';
        return $errors;
    }
    
    // بررسی اندازه فایل
    if ($file['size'] > $max_size) {
        $errors[] = 'حجم فایل نباید بیشتر از ' . round($max_size / 1048576, 2) . ' مگابایت باشد';
    }
    
    // بررسی نوع فایل
    $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($file_ext, $allowed_types)) {
        $errors[] = 'فقط فایل‌های ' . implode(', ', $allowed_types) . ' مجاز هستند';
    }
    
    // بررسی MIME Type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    $allowed_mime_types = [
        'pdf' => 'application/pdf'
    ];
    
    if (!in_array($mime_type, $allowed_mime_types)) {
        $errors[] = 'نوع فایل نامعتبر است';
    }
    
    return $errors;
}

// ایجاد نام فایل امن
function generate_secure_filename($original_name) {
    $extension = pathinfo($original_name, PATHINFO_EXTENSION);
    $name = pathinfo($original_name, PATHINFO_FILENAME);
    
    // حذف کاراکترهای غیرمجاز
    $name = preg_replace('/[^a-zA-Z0-9_\x{0600}-\x{06FF}\-]/u', '', $name);
    
    // اضافه کردن timestamp و random string
    $secure_name = uniqid() . '_' . time() . '_' . substr(md5($name), 0, 8) . '.' . $extension;
    
    return $secure_name;
}

// جلوگیری از Hotlink
function prevent_hotlink() {
    $allowed_domains = ['localhost', 'jovehhub.ir', 'www.jovehhub.ir'];
    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    
    if (!empty($referer)) {
        $referer_domain = parse_url($referer, PHP_URL_HOST);
        if (!in_array($referer_domain, $allowed_domains)) {
            header('HTTP/1.0 403 Forbidden');
            die('دسترسی غیرمجاز');
        }
    }
}

// محدود کردن دسترسی به فایل‌ها
function protect_file($file_path) {
    // بررسی مسیر
    if (strpos($file_path, '..') !== false) {
        die('مسیر نامعتبر');
    }
    
    // بررسی نوع فایل
    $allowed_extensions = ['pdf', 'png', 'jpg', 'jpeg', 'gif'];
    $extension = pathinfo($file_path, PATHINFO_EXTENSION);
    
    if (!in_array(strtolower($extension), $allowed_extensions)) {
        die('نوع فایل مجاز نیست');
    }
    
    // بررسی وجود فایل
    if (!file_exists($file_path)) {
        header('HTTP/1.0 404 Not Found');
        die('فایل یافت نشد');
    }
}

// Rate Limiting
class RateLimiter {
    private $limit;
    private $time_window;
    private $storage_path;
    
    public function __construct($limit = 100, $time_window = 3600) {
        $this->limit = $limit;
        $this->time_window = $time_window;
        $this->storage_path = __DIR__ . '/../cache/rate_limiting/';
        
        if (!is_dir($this->storage_path)) {
            mkdir($this->storage_path, 0755, true);
        }
    }
    
    public function check($ip) {
        $filename = $this->storage_path . md5($ip) . '.json';
        
        if (file_exists($filename)) {
            $data = json_decode(file_get_contents($filename), true);
            
            if (time() - $data['timestamp'] < $this->time_window) {
                if ($data['count'] >= $this->limit) {
                    return false;
                }
                $data['count']++;
            } else {
                $data = ['count' => 1, 'timestamp' => time()];
            }
        } else {
            $data = ['count' => 1, 'timestamp' => time()];
        }
        
        file_put_contents($filename, json_encode($data));
        return true;
    }
}

// جلوگیری از حملات CSRF
function csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verify_csrf_token($token) {
    if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
        die('توکن CSRF نامعتبر است');
    }
}

// جلوگیری از حملات Brute Force
class BruteForceProtection {
    private $max_attempts = 5;
    private $lockout_time = 900; // 15 دقیقه
    private $storage_path;
    
    public function __construct() {
        $this->storage_path = __DIR__ . '/../cache/bruteforce/';
        
        if (!is_dir($this->storage_path)) {
            mkdir($this->storage_path, 0755, true);
        }
    }
    
    public function check($ip) {
        $filename = $this->storage_path . md5($ip) . '.json';
        
        if (file_exists($filename)) {
            $data = json_decode(file_get_contents($filename), true);
            
            if (time() - $data['timestamp'] < $this->lockout_time) {
                if ($data['attempts'] >= $this->max_attempts) {
                    return false;
                }
            } else {
                $data = ['attempts' => 0, 'timestamp' => time()];
            }
        } else {
            $data = ['attempts' => 0, 'timestamp' => time()];
        }
        
        return true;
    }
    
    public function record_failure($ip) {
        $filename = $this->storage_path . md5($ip) . '.json';
        
        if (file_exists($filename)) {
            $data = json_decode(file_get_contents($filename), true);
            
            if (time() - $data['timestamp'] < $this->lockout_time) {
                $data['attempts']++;
            } else {
                $data = ['attempts' => 1, 'timestamp' => time()];
            }
        } else {
            $data = ['attempts' => 1, 'timestamp' => time()];
        }
        
        file_put_contents($filename, json_encode($data));
    }
    
    public function record_success($ip) {
        $filename = $this->storage_path . md5($ip) . '.json';
        
        if (file_exists($filename)) {
            unlink($filename);
        }
    }
}

// ایجاد تصویر کپچا
function generate_captcha() {
    $captcha_code = substr(str_shuffle('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 6);
    $_SESSION['captcha_code'] = $captcha_code;
    
    $image = imagecreatetruecolor(150, 50);
    $bg_color = imagecolorallocate($image, 255, 255, 255);
    $text_color = imagecolorallocate($image, 0, 0, 0);
    $line_color = imagecolorallocate($image, 200, 200, 200);
    
    imagefilledrectangle($image, 0, 0, 150, 50, $bg_color);
    
    // خطوط نویز
    for ($i = 0; $i < 10; $i++) {
        imageline($image, rand(0, 150), rand(0, 50), rand(0, 150), rand(0, 50), $line_color);
    }
    
    // نوشتن متن
    $font = __DIR__ . '/../assets/fonts/Vazirmatn-Regular.ttf';
    if (file_exists($font)) {
        imagettftext($image, 20, rand(-10, 10), 20, 35, $text_color, $font, $captcha_code);
    } else {
        imagestring($image, 5, 50, 15, $captcha_code, $text_color);
    }
    
    ob_start();
    imagepng($image);
    $image_data = ob_get_clean();
    imagedestroy($image);
    
    return 'data:image/png;base64,' . base64_encode($image_data);
}

// اعتبارسنجی کپچا
function validate_captcha($user_input) {
    return isset($_SESSION['captcha_code']) && 
           strtolower($user_input) === strtolower($_SESSION['captcha_code']);
}
?>